%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% The following script solves the nonlinear equations of %%%%
%%%% motion of the elastica constrained by an oscillating   %%%%
%%%% sliding sleeve. For more information see the main      %%%%
%%%% publication:                                           %%%%
%%%% "Stabilization against gravity and self-tuning of an   %%%%
%%%%  elastic variable-length rod through an oscillating    %%%%
%%%%  sliding sleeve"                                       %%%%
%%%%  P.Koutsogiannakis, D. Misseroni, D. Bigoni,           %%%%
%%%%  F. Dal Corso                                          %%%%
%%%% Software author: P. Koutsogiannakis                    %%%%
%%%% Date: 6/4/2023                                         %%%%
%%%% Reqiured files:                                        %%%%
%%%%    - elastica_system.m                                 %%%%
%%%%    - elastica_system_small_deflections.m               %%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


clear all
close all
set(0, 'DefaultFigureRenderer', 'painters'); % for compatibility with systems without opengl


%% ---- System parameters ---- %%

F    = 32 % input frequency in Hz
ug   = 5 % amplitude of the input motion in mm
m    = 0.19697 % value of the total mass of the system in kg
B    = 1.4363 % stiffness of the rod in N*m^2
g    = 9.81 % acceleration of gravity in m/s^2
Lmax = 10 % maximum length allowed outside the sliding sleeve in meters

mu = 0 % dimensionless friction coefficient, inside the sliding sleeve
ze = 0.005 % dimensionless viscous damping coefficient, lumped mass

SPC   = 80 % Steps per Cycle, number of timesteps in one period of the input oscillation
t_end = 10 % final integration time to terminate the simulation in seconds
output_filename = 'out.mat' % name for the output .mat file


% Initial values
L0  = 0.105 % initial length of the rod outside the sliding sleeve in meters
vx0 = 2 % initial velocity of the lumped mass in the horizontal direction


isLinear = false % binary flag, true: use a linearized version of the system, false: use the elastica equations
plot_simulation = true % binary flag for plotting the simulation state, set to false if not necessary
update_plot_timesteps = 20 % number of timesteps between updates to the plot


%% ---- Time-integration of the equations ---- %%

% set a datastructure holding the system parameters
Uinit = [ 0; L0; vx0; 0; L0; 0; pi/2; 0 ]; % initial value of the vector holding the degrees of freedom
omega = F*2*pi; % angular frequency of the input oscillation

params.m = m;
params.B = B;
params.ze = ze;
params.mu = mu;
params.g = g;
params.F = F;
params.ug = ug;
params.ref    = @(t) -ug/1000            * sin(2*pi*F * t); % input oscillation function
params.ref_t  = @(t) -ug/1000*(2*pi*F)   * cos(2*pi*F * t); % velocity input oscillation function
params.ref_tt = @(t)  ug/1000*(2*pi*F)^2 * sin(2*pi*F * t); % acceleration of input oscillation function

params.SPC = SPC;

options = optimoptions('fsolve', 'Display','off', 'UseParallel',false, 'MaxFunctionEvaluations',10000, 'MaxIterations',1000, 'FunctionTolerance',1e-6); % fsolve options




dt = 2*pi/omega / SPC; % timestep length

t(1) = 0; % time value for the timestep
Usol(:,1) = Uinit; % Usol holds the state of the system for every tiemstep calculated

if plot_simulation
    fig = figure();
end

n = 0; % the index of the timestep
U0 = Usol(:,1);
disp(['n     |   x          y          vx         vy         el         th         be         vl'])
while t(end) < t_end
    n = n+1;
    t(n+1) = t(n) + dt;

    if abs(Usol(6,n)) > 1e-3 && ~isLinear % calculate using the elastica equations when the deflection angle is not close to zero
        func = @(Uvar) elastica_system(t(n+1), Uvar, Usol(:,n), dt, params);
        U0 = Usol(:,n); U0(1) = U0(1)+dt*U0(3); U0(2) = U0(2)+dt*U0(4);
        [Usol(:,n+1),Fval,exitflag,output] = fsolve(func,U0, options);
    else % calculate using the small deflection linearized elastica equations
        Uprev = Usol( [1:end-2,end] ,n);
        func = @(Uvar) elastica_system_small_deflections(t(n+1), Uvar, Uprev, dt, params);
        [U_aux,Fval,exitflag,output] = fsolve(func,Uprev, options);
        Usol(:,n+1) = [U_aux(1:end-1); pi/2; U_aux(end)];
    end

    % print state of the system to the screen
    fprintf(1,'%04d  |  ',n);
    if Usol(1,n+1)>=0
        fprintf(1,' ');
    end
    fprintf(1,num2str((Usol(:,n+1))','  %9.3f'));
    fprintf(1,'\n');

    % plot simulation state
    if plot_simulation && ~mod(n,update_plot_timesteps)
        plot_simulation_state(fig, t, Usol, params);
    end

    if Usol(5,n+1) < 1e-4 || Usol(5,n+1) > Lmax % terminate simulation if the rod is fully injected of the max external length is reached
        break;
    end
end
save(output_filename, 't', 'Usol', 'params')

function plot_simulation_state(fig, t, Usol, params)
    n = length(t)-1;
    SPC = params.SPC;
    figure(fig);
    clf();

    subplot(2,2,[1,3]); % plot lumped mass trajectory
    hold on
    title(['t = ',num2str(t(n+1)),' sec,  F=',num2str(params.F),' ,  \zeta=',num2str(params.ze),' ,  \mu=',num2str(params.mu)])
    xline(0)
    periods = 40;
    xx = Usol(1,max([1,(end-periods*SPC)]):end) * cos(0) + Usol(2,max([1,(end-periods*SPC)]):end) * (sin(0));
    yy = Usol(1,max([1,(end-periods*SPC)]):end) * (-sin(0)) + Usol(2,max([1,(end-periods*SPC)]):end) * (cos(0));
    plot(xx, yy);
    plot(xx(end), yy(end), '.', 'markersize', 10);
    % xlim([-1, 1]);
    ylim([-0.5, 1]);
    axis equal

    subplot(2,2,2) % plot external length timeseries
    plot(t,Usol(5,:))
    xlabel('t')
    ylabel('ell')

    subplot(2,2,4) % plot deflection timeseries
    plot(t,Usol(6,:))
    xlabel('t')
    ylabel('\theta')

    drawnow
end